﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Web.UI;
using System.Web.UI.WebControls;

using VIRP_WEB.BaseCode;
using VIRP.EFR.BLL;
using VIRP.EFR.BO;

namespace VIRP_WEB.Custom.EFR
{
    public partial class PatientImaging : ImagingBasePage
    {
        /// <summary>
        /// https://stackoverflow.com/questions/8114941/encoded-url-does-not-work
        /// </summary>
        private static Dictionary<string, string> EncodingReservedCharacters = new Dictionary<string, string>
        {
            { "%21", "!" },
            { "%2A", "*" },
            { "%27", "'" },
            { "%28", "(" },
            { "%29", ")" },
            { "%3B", ";" },
            { "%3A", ":" },
            { "%40", "@" },
            { "%26", "&" },
            { "%3D", "=" },
            { "%2B", "+" },
            { "%24", "$" },
            { "%2C", "," },
            { "%2F", "/" },
            { "%3F", "?" },
            { "%23", "#" },
            { "%5B", "[" },
            { "%5D", "]" }
        };

        private readonly PatientImagingManager ImagingManager;

        private int PatientId;

        public PatientImaging()
        {
            ImagingManager = new PatientImagingManager();
        }

        protected override void Page_Load(object sender, EventArgs e)
        {
            ReplacePageTitle();

            if (ViewState[OriginallyNavigatedFrom] == null)
                ViewState[OriginallyNavigatedFrom] = Request.UrlReferrer;

            if (!IsPostBack)
            {
                var patientSummary = GetOrStorePatientSummary();
                if (patientSummary != null)
                {
                    LabelPatientName.Text += patientSummary.FullName;
                    LabelPatientSnum.Text += patientSummary.Snum;

                    ReplacePageTitle("Patients > Patient Lookup > Imaging > " + patientSummary.FullName);

                    LoadData(patientSummary.Id, patientSummary.Icn);

                    GetPatientImageData(patientSummary.Id, patientSummary.Icn);
                }
            }

            AddFormApplicationRole();

            RegistryContext.Value = CurrentUser;
            RegistryId.Value = CurrentRegistryId.ToString();

            ClientScript.RegisterStartupScript(this.GetType(), "checkImagingStatus", "<script type='text/javascript'>checkImagingStatus();</script>");
        }

        private void GetPatientImageData(int patientId, string patientIcn)
        {
            ObjectDataSourcePatientsImages.SelectParameters["PATIENT_ID"].DefaultValue = patientId.ToString();
            ObjectDataSourcePatientsImages.SelectParameters["PATIENT_ICN"].DefaultValue = patientIcn;
            ObjectDataSourcePatientsImages.SelectParameters["CURRENT_USER"].DefaultValue = CurrentUser;
            ObjectDataSourcePatientsImages.SelectParameters["CURRENT_REGISTRY_ID"].DefaultValue = CurrentRegistryId.ToString();
            ObjectDataSourcePatientsImages.Select();
        }

        private void LoadData(int patientId, string patientIcn)
        {
            IMAGE_ANALYSIS patientImageAnalysis = ImagingManager.GetPatientImageAnalysis(CurrentUser, CurrentRegistryId, patientIcn);
            if (patientImageAnalysis != null)
            {
                // ImagesInCprsFlag
                if (patientImageAnalysis.IMAGES_IN_CPRS_FLAG.HasValue)
                    DropDownListInCprs.SelectedIndex = patientImageAnalysis.IMAGES_IN_CPRS_FLAG.Value ? 1 : 2;
                else
                {
                    if (HasImages(patientId))
                        DropDownListInCprs.SelectedIndex = 0;
                    else
                        DropDownListInCprs.SelectedIndex = 1;
                }

                // FragmentVerifiedFlag
                if (patientImageAnalysis.FRAGMENT_VERIFIED_FLAG.HasValue)
                {
                    DropDownListFragmentVerified.SelectedIndex = patientImageAnalysis.FRAGMENT_VERIFIED_FLAG.Value ? 1 : 2;
                    LabelSummaryFragmentVerified.Text += DropDownListFragmentVerified.Items[DropDownListFragmentVerified.SelectedIndex].Text;
                }
                else
                    DropDownListFragmentVerified.SelectedIndex = 0;

                // BoneFractureDocumentedFlag
                if (patientImageAnalysis.BONE_FRACTURE_DOCUMENTED_FLAG.HasValue)
                    DropDownListBoneFracture.SelectedIndex = patientImageAnalysis.BONE_FRACTURE_DOCUMENTED_FLAG.Value ? 1 : 2;
                else
                    DropDownListBoneFracture.SelectedIndex = 0;

                // FragmentsInJointSpaceFlag
                if (patientImageAnalysis.FRAGMENTS_IN_JOIN_SPACE_FLAG.HasValue)
                    DropDownListJointSpace.SelectedIndex = patientImageAnalysis.FRAGMENTS_IN_JOIN_SPACE_FLAG.Value ? 1 : 2;
                else
                    DropDownListJointSpace.SelectedIndex = 0;

                // StandardImageFragmentSizeId
                if (patientImageAnalysis.STD_IMAGE_FRAGMENT_SIZE_ID.HasValue)
                    DropDownListFragmentSize.Items[patientImageAnalysis.STD_IMAGE_FRAGMENT_SIZE_ID.Value].Selected = true;
                else
                    DropDownListFragmentSize.SelectedIndex = 0;

                // JointSpace
                if (!string.IsNullOrEmpty(patientImageAnalysis.JOINT_SPACE))
                    TextBoxJoint.Text = patientImageAnalysis.JOINT_SPACE;
                else
                    TextBoxJoint.Text = string.Empty;

                // OtherObservations
                if (!string.IsNullOrEmpty(patientImageAnalysis.OTHER_OBSERVATIONS))
                    TextBoxOtherObservations.Text = patientImageAnalysis.OTHER_OBSERVATIONS;
                else
                    TextBoxOtherObservations.Text = string.Empty;

                // DateReviewed
                if (patientImageAnalysis.DATE_REVIEWED.HasValue)
                    TextBoxDateImagingReviewedByStaff.Text = patientImageAnalysis.DATE_REVIEWED.HasValue
                        ? patientImageAnalysis.DATE_REVIEWED.Value.ToString(DateFormat) : string.Empty;
                else
                    TextBoxDateImagingReviewedByStaff.Text = string.Empty;

                // DateImagingReviewedByStaff
                if (patientImageAnalysis.DATE_REVIEWED.HasValue)
                    LabelSummaryDateImagingReviewedByStaff.Text += patientImageAnalysis.DATE_REVIEWED.Value.ToString(DateFormat);

                HiddenFieldHasHistory.Value = "1";
            }
            else
                ClearFields();
        }

        private void ClearFields()
        {
            // ImagesInCprsFlag
            DropDownListInCprs.SelectedIndex = 0;

            // FragmentVerifiedFlag
            DropDownListFragmentVerified.SelectedIndex = 0;

            // BoneFractureDocumentedFlag
            DropDownListBoneFracture.SelectedIndex = 0;

            // FragmentsInJointSpaceFlag
            DropDownListJointSpace.SelectedIndex = 0;

            // StandardImageFragmentSizeId
            DropDownListFragmentSize.SelectedIndex = 0;

            // JointSpace
            TextBoxJoint.Text = string.Empty;

            // OtherObservations
            TextBoxOtherObservations.Text = string.Empty;

            // DateReviewed
            TextBoxDateImagingReviewedByStaff.Text = string.Empty;

            // DateImagingReviewedByStaff
            LabelSummaryDateImagingReviewedByStaff.Text += string.Empty;

            HiddenFieldHasHistory.Value = "0";
        }

        protected void SaveButton_Click(object sender, EventArgs e)
        {
            if (Page.IsValid && AuthorizedUser())
            {
                var patientSummary = GetOrStorePatientSummary();

                IMAGE_ANALYSIS objSave = GetImageAnalysisFromFields(patientSummary.Id);

                ImagingManager.InsertUpdateImageAnalysis(CurrentUser, CurrentRegistryId, objSave);

                SetSaveSuccessMessage();

                LoadData(patientSummary.Id, patientSummary.Icn);
            }
        }

        protected void DeleteButton_Click(object sender, EventArgs e)
        {
            var patientSummary = GetOrStorePatientSummary();

            if (ImagingManager.DeleteImageAnalysis(CurrentUser, CurrentRegistryId, patientSummary.Icn))
                SetSaveMessage("Image Summary Deleted.");
            else
                SetSaveMessage("Error attempting to deleted Image Summary.");

            LoadData(patientSummary.Id, patientSummary.Icn);
        }

        private IMAGE_ANALYSIS GetImageAnalysisFromFields(int patientId)
        {
            return new IMAGE_ANALYSIS
            {
                PATIENT_ID = patientId,
                BONE_FRACTURE_DOCUMENTED_FLAG = GetDropDownListBooleanValue(DropDownListBoneFracture),
                FRAGMENTS_IN_JOIN_SPACE_FLAG = GetDropDownListBooleanValue(DropDownListJointSpace),
                FRAGMENT_VERIFIED_FLAG = GetDropDownListBooleanValue(DropDownListFragmentVerified),
                IMAGES_IN_CPRS_FLAG = GetDropDownListBooleanValue(DropDownListInCprs),
                STD_IMAGE_FRAGMENT_SIZE_ID = GetDropDownSelectedIndex(DropDownListFragmentSize),
                JOINT_SPACE = GetStringValueFromTextBox(TextBoxJoint),
                OTHER_OBSERVATIONS = GetStringValueFromTextBox(TextBoxOtherObservations),
                DATE_REVIEWED = GetDateTimeValueFromTextBox(TextBoxDateImagingReviewedByStaff)
            };
        }

        private bool HasImages(int patientId)
        {
            var sessionKey = string.Format("ImagesForPatientCount|{0}", patientId);

            var tryGetCount = Helpers.SessionItemValueGet(Session[sessionKey]);
            if (!string.IsNullOrEmpty(tryGetCount))
                return int.Parse(tryGetCount) > 0;

            var imageCount = ImagingManager.GetImagesForPatientCount(patientId, CurrentUser, CurrentRegistryId);
            Session[sessionKey] = SessionItemFactory.Create(imageCount);
            return imageCount > 0;
        }

        protected string GetBodyPartsImaged(object patientImageId)
        {
            var id = Convert.ToInt32(patientImageId);

            return ImagingManager.GetBodyPartsImaged(CurrentUser, CurrentRegistryId, id);
        }

        protected void GridImages_DataBound(object sender, EventArgs e)
        {
            // ImagesInCprsFlag
            if (DropDownListInCprs.SelectedIndex == 0)
            {
                var patientSummary = GetOrStorePatientSummary();
                if (HasImages(patientSummary.Id))
                    DropDownListInCprs.SelectedIndex = 0;
                else
                    DropDownListInCprs.SelectedIndex = 1;
            }
        }

        protected void LinkButtonPatientInformation_Click(object sender, EventArgs e)
        {
            if (int.TryParse(Request.QueryString["patientId"], out this.PatientId))
                Response.Redirect("~/Custom/EFR/Patients/Patient.aspx?PatientId=" + this.PatientId, false);
        }

        protected void LinkButtonWorkflows_Click(object sender, EventArgs e)
        {
            if (int.TryParse(Request.QueryString["patientId"], out this.PatientId))
                Response.Redirect("~/Custom/EFR/Patients/PatientWorkflows.aspx?PatientId=" + this.PatientId, false);
        }

        protected void LinkButtonRelatedDiagnoses_Click(object sender, EventArgs e)
        {
            if (int.TryParse(Request.QueryString["patientId"], out this.PatientId))
                Response.Redirect("~/Custom/EFR/Patients/PatientRelatedDiagnoses.aspx?PatientId=" + this.PatientId, false);
        }

        protected void LinkButtonLabTests_Click(object sender, EventArgs e)
        {
            if (int.TryParse(Request.QueryString["patientId"], out this.PatientId))
                Response.Redirect("~/Custom/EFR/Patients/PatientRelatedLabTests.aspx?PatientId=" + this.PatientId, false);
        }

        protected string GetDetailsLink(object patientId, object patientImageId)
        {
            const string BaseLink = "PatientImageDetails.aspx?patientId={0}&patientImageId={1}";

            return string.Format(BaseLink, Helpers.AntiXssEval(Convert.ToString(patientId)), Helpers.AntiXssEval(Convert.ToString(patientImageId)));
        }

        protected void GridImages_RowDataBound(object sender, GridViewRowEventArgs e)
        {
            if (e.Row.RowType == DataControlRowType.DataRow && e.Row.DataItem is PatientImageView)
            {
                PatientImageView row = (PatientImageView)e.Row.DataItem;
                HandleLink(e, row.VIEWER_URL, "ImageViewer");
                HandleLink(e, row.REPORT_URL, "ReportViewer");
            }
        }

        private void HandleLink(GridViewRowEventArgs e, string url, string name)
        {
            const string HyperLinkBase = "HyperLink{0}";
            const string LabelBase = "Label{0}";

            HyperLink link = (HyperLink)e.Row.FindControl(string.Format(HyperLinkBase, name));
            link.ToolTip = "Click to launch image viewer.";

            Label label = (Label)e.Row.FindControl(string.Format(LabelBase, name));
            label.ToolTip = "Click to see report.";

            link.NavigateUrl = GetValidatedImagingLink(Helpers.AntiXssEval(url));
        }

        private string GetValidatedImagingLink(string url)
        {
            const char Percent = '%';

            var result = new StringBuilder();

            for (int i = 0; i < url.Length; i += 1)
            {
                if (url[i].Equals(Percent))
                {
                    var word = url.Substring(i, 3);
                    if (EncodingReservedCharacters.ContainsKey(word))
                        word = EncodingReservedCharacters[word];

                    result.Append(word);
                    i += 2;
                }
                else
                    result.Append(url[i]);
            }

            return result.ToString();
        }
    }
}
